;FILE: NEWQPSK.ASM, Last edition: 22nd August 1995
;new design for the multi-tone, QPSK modem.
;(c) 1995 Pawel Jalocha, SP9VRC,
;e-mail: jalocha@chopin.ifj.edu.pl, sp9vrc@gw.sp9kby.ampr.org
;
;Features:
; - 15 carriers spaced by 125 Hz modulated with DQPSK
;   (differential 4-level phase keying) at 83.33 baud
; - Total raw data rate 2500 bps
; - Two phase preamble for frequency shift correction
;   and fast symbol sync.
; - Forward Error Correction: two simple schemes with different levels
;   of data redundancy
; - Time/frequency diversity (interleave) to de-localize burst errors
;
;This software is not to be used for purposes other than amateur radio
;without a written permision of the author.
;Usage for profit or in commercial/military products is explicitly prohibited.

        nolist
	include 'leonid'
	list

        scsjmp short    ;force short jumps for .if/.endi, .while/.endw, etc.

        title 'New multi-tone QPSK modem by SP9VRC'

;*************************************************************************
;compile-time constants

EVM56K  equ 0           ;0 => DSPCARD4
                        ;1 => EVM56002

SampleFreq equ 8000.0

DecimateRatio equ 3        ;decimation ratio
                           ;the maximum bandwidth of the anti-alias filter:
                           ;sampled bandwidth = 8000/3 Hz

AliasFilterLen equ 64           ;the length of the anti-alias FIR filter

WindowLen       equ 64          ;sliding-window length
WindowLenLog    equ 6           ;and its 2-base logarythm
                                ; => FT freq. bin = 8000/3/64 Hz
SymbolLen       equ WindowLen/2 ;the length (time-spacing) of our symbols
                                ; => symbol rate = 8000/3/32 = 83.3333 baud

FirstDataCarr   equ 18          ;first data carrier at 18*(8000/3/64) = 750 Hz
DataCarrSepar   equ 3           ;data carriers separated by 3*(8000/3/64) = 125 Hz
                                ;(a very fixed parameter)
DataCarriers    equ 15          ;15 data carriers
                                ; => total bandwidth = 15*3*(8000/3/64) = 1875 Hz
                                ; => total data rate = 15*83.3333 = 2500 bps

FirstTuneCarr   equ FirstDataCarr+DataCarrSepar
                                       ;first carrier for tune/sync. at 875 Hz
TuneCarrSepar   equ DataCarrSepar*4    ;tune carriers separated by 4*125 Hz
TuneCarriers    equ (DataCarriers+1)/4 ;4 tune carriers

RxPipeLen       equ 4           ;pipe for auto-tuner, differential phase
                                ;decoder, bit synchronizer.

TxMinIdle       equ 50 ;[symbols]  minimal idling time should a new
                                ;packet come just at the end of a transmition
TxTuneLen       equ 32 ;[symbols]  the time for which the tuning sequence
                                ;is transmitted: 32/100 baud = 0.32 sec
                                ;make this larger if your radio has long
                                ;Tx switch-on time.
TxSyncLen       equ 32 ;[symbols] the time for which the symbol-sync sequence
                                ;is transmitted: 32/100 baud = 0.32 sec
TxPreData       equ 4  ;[symbols] send that many 0 symbols before the user data
TxPostData      equ 2  ;[symbols] and that many after the user data
TxJamLen        equ 16 ;[symbols] the time for which the jamming sequence
                                ;is transmitted (for faster DCD switch-off).
TxAttenuate     equ  20.0 ;[dB] attenuate the output CODEC's signal.
                                ;Set this between 0.0 and 90.0
                                ;to accomodate the output level for the
                                ;modulator input of your tranceiver.
TxPTThack       equ 1           ;Do my own persistance decision before
                                ;pushing the PTT. This is done because
                                ;LEONID's PTT logic doesn't fit well
                                ;the specific needs of this modem.
TxPersistance   equ 0.25        ;Persistance level per slot-time
                                ;slot-time equals to symbol length

RxGain          equ 10.0 ;[dB] Rx CODEC's input gain
                                ;set this between 0 and 22.5 dB depending on
                                ;the audio level from your tranceiver
RxAGC           equ 1           ;enable the automatic gain adjustment
                                ;for the CODEC's audio input.
                                ;ideally you should avoid the AGC and set
                                ;RxGain for best audio level match.
                                ;Note, that the AGC may get fooled up by
                                ;for example long periods of total silence,
                                ;thus you should rather keep the quelch (if any)
                                ;open all the time.
RxAGC_MSfollow  equ 1.0/16.0    ;update weight per half-symbol
                                ;for the Mean Square tracking
RxAGChold       equ 320 ;[half-symbol]  AGC peak hold time
RxAGCfall       equ 32  ;[half-symbol]  AGC falling period/1.5 dB step
RxAGCminMS      equ 0.008       ;min. and max. tolerable audio Mean Square level
RxAGCmaxMS      equ 0.04
RxAGCmaxPeak    equ 0.75        ;max. allowed peak audio level
RxAGCsensePeak  equ 0           ;take (or don't take) into acount the peak level
                                ;when deciding about the gain changes
                                ;taking the peak into account makes the AGC
                                ;more "jumpy" which is not good in my opinion.
RxAGCavoidDCD   equ 0           ;disallow gain changes when DCD is on

RxAverFollow    equ 4           ;follow factor for averaging the correlations
                                ;during the idle and tune phase.
                                ;the actuall weight is 1/2^RxAverFollow
                                ;per half-symbol
RxMinTune       equ 16 ;[symbols] minimal duration of the detected tune tones
                                ;(a given set of pure tones)
                                ;before deciding on the freq. error
                                ;and going into the "tune" phase.
RxTuneTimeout   equ 56 ;[symbols] timeout for the "tune" phase. If bit-sync.
                                ;is not found for that many symbols
                                ;sync. attempt is aborted
RxMinSync       equ 16 ;[symbols] minimal duration of the detected sync.
                                ;(alternating phase carriers)
                                ;before deciding on the symbol timing
                                ;and going into the "data" phase.
RxUpdateHold    equ 10  ;[symbols] hold the Rx DCD and carrier power
                                ;update after the symbol sync. has been detected.
DCDTuneAverWeight equ 1.0/16.0  ;Weight for averaging of DCD and tuning
                                ;stat. during data phase
DCDThreshold    equ 0.020       ;DCD threshold: higher number -> the DCD
                                ;is more tolerant to noise.
                                ;0.02 is real close to the noise limit
DCDMaxDrop      equ 12 ;[symbols] DCD may drop for that many symbols
                                ;before it is assumed to be gone for good
RxDataSyncFollow equ 5          ;follow weight for the carrier powers
                                ;and sync. correction while in the data phase
RxFreqFollowWeight equ 1.0/32   ;weight/symbol to follow the frequency error
                                ;during the data phase
RxEarlyDCD      equ 1           ;set DCD to on already in Idle state
                                ;when some signs of the tune preamble
                                ;are being seen. This speeds up the channel
                                ;busy detection and makes less colisions.
RxCtrlUpDown    equ 1           ;Control the up/down switches to follow
                                ;automatically frequency drift.
                                ;this conflicts with RxMonitorRxState !
UpDownPulseLen  equ 10 ;[symbols] Pulse length for the Up/Down
                                ;corrections
                                ;10 = 10/100 = 0.1 sec.
UpDownCorrThres equ 20.0 ;[Hz] Up/Down corrections will be done only
                                ;if frequency error is larger than this
UpDownReverse   equ 0           ;reverse the sense: usefull for LSB operation.

RightChannel    equ 0           ;Use the right CODEC's channel not the left.
                                ;I didn't try this but Timo says it works.

FEC     equ 3   ;0 => no Forward Error Correction
                ;
                ;1 => one-bit FEC: for every 11 bits you add 4 so afterwards
                ;     single bit errors can be detected and corrected.
                ;     The effective data rate falls down to 2500*11/15 = 1833.33 bps
                ;
                ;3 => Walsh function FEC: 5 data bits -> 15 bits.
                ;     Receiver can correct up to 3 bit errors.
                ;     The effective data rate falls down to 2500*5/15 = 833.33 bps

FECflickerRedLED equ 1  ;flicker the red LED if errors are seen
                        ;in the FEC decoder.

Interleave equ 8        ;interleave factor (0 = interleave disabled)
                        ;bits from one bit-batch are transmitted
                        ;at different times so a one-symbol burst error
                        ;will corrupt only one bit from a bit-batch.
                        ;The larger the interleave the longer burst errors
                        ;can possibly be recovered (but at same time you
                        ;prolong the TxFlush phase).
                        ;For a given Interleave every transmition is prolonged
                        ;by Interleave/2*DataCarriers symbols
                        ;Note that Interleave makes sense when you use FEC
                        ;at same time, otherwise it only prolongs the packets
                        ;without any benefits.

ScrambledInterleave equ 1 ;0 => linear interleave
                          ;1 => scrambled interleavve to maximize the
                          ;     time/freq. distance between bits belonging
                          ;     to same FEC bit-batch

;the FEC and Interleave code is written exclusively for DataCarriers=15

        if FEC==0
BitBatchLen     equ DataCarriers
        endif
        if FEC==1
BitBatchLen     equ 11
        endif
        if FEC==3
BitBatchLen     equ 5
        endif

; The overall frequency plan is the following: 15 carriers
; carrier #0 at 750 Hz, #7 (the middle one) at 1625 Hz, #14 at 2500 Hz
; absolute maximum allowed mistune is +/- 80 Hz,
; but we rather expect up to +/- 60 Hz.

BufLen   equ      512              ;sample buffer length
                                   ;with some safety margin

SPY      equ 0  ; SPY=0 => KISS mode (_all_ SPY_ sub-options must be set to 0 !)
                ; SPY=1 => KISS code excluded, selected debug code included

SPY_RxIQ        equ 0
SPY_RxWindow    equ 0
SPY_RxFFT       equ 0
SPY_PeakMS      equ 0
SPY_CODECgain   equ 0
SPY_RxCarIQ     equ 0
SPY_RxCorrel    equ 0
SPY_RxStateRou  equ 0
SPY_RxCarFreq   equ 0
SPY_RxToneTune  equ 0
SPY_RxSyncTune  equ 0
SPY_RxDataIQ    equ 0
SPY_RxCrossedIQ equ 0
SPY_RxCrossedPhase equ 0
SPY_RxCrossedPhaseAver equ 0
SPY_RxPhaseAbsAver equ 0
SPY_DCDnum       equ 0
SPY_DCDaver      equ 0
SPY_DataTuneAver equ 0
SPY_DataSyncAver equ 0
SPY_RxDataWord   equ 0

MonitorRxState   equ 0          ;monitor the state of the Rx by flashing LEDs
                                ;connected to the UP, DOWN, CAT lines of the
                                ;left radio channel. This is more for me to
                                ;debug the code.
                                ;This option conflicts with RxCtrlUpDown
                                ;when you use the left channel.

MonitorRxAGC    equ 0           ;monitor the gain changes down by the AGC
                                ;with the UP and DOWN LEDs
                                ;This option conflicts with RxCtrlUpDown
                                ;when you use the left channel.

;*************************************************************************
;Some simple macros

      if RightChannel
PTT     macro mode      ;PTT line: clr/set/chg
        b\mode #4,X:$FFE4
        endm
      else
PTT     macro mode      ;PTT line: clr/set/chg
        b\mode #0,X:$FFE4
;      b\mode #3,X:$FFE4       ;*** DEBUG ***
        endm
      endif

      if RightChannel
PushUp  macro mode      ;Push UP line of the TRX
        b\mode #5,X:$FFE4
        endm
      else
PushUp  macro mode
        b\mode #1,X:$FFE4
        endm
      endif

      if RightChannel
PushDown macro mode      ;Push DOWN line of the TRX
        b\mode #6,X:$FFE4
        endm
      else
PushDown macro mode
        b\mode #2,X:$FFE4
        endm
      endif

UpLED   macro mode      ;UP line (a red LED connected)
        b\mode #1,X:$FFE4
        endm

DownLED macro mode      ;DOWN line (a red LED connected)
        b\mode #2,X:$FFE4
        endm

YellowLED macro mode    ;CAT line (a yellow LED connected)
        b\mode #3,X:$FFE4
        endm

RedLED  macro mode      ; Red LED clr/set/chg
        b\mode #13,X:$FFE4
        endm

LongDecayAver macro Input,Aver,Scale
              sub Aver,Input
              rep Scale
                asr Input
              add Input,Aver
              endm

;*************************************************************************
;The actuall code (internal/external program RAM)

        LOMEM P:$0000
        HIMEM P:$1FFF

        org     p:user_code

        jmp <Initialize

DoFFT   ;routine to perform a complex FFT (and Inverse FFT too)
        ;copied from Motorola's Dr. BuB.
        ;alters a,b,x,y, rnm0, rnm1, n2, rnm4, rnm5, rnm6
        ;Uses 6 locations on System Stack

         move    #WindowLen/2,n0   ;initialize butterflies per group
         move    #1,n2             ;initialize groups per pass
         move    #WindowLen/4,n6   ;initialize C pointer offset
         move    #-1,m0            ;initialize A and B address modifiers
         move    m0,m1             ;for linear addressing
         move    m0,m4
         move    m0,m5
         move    #0,m6             ;initialize C address modifier for
                                   ;reverse carry (bit-reversed) addressing
;
; Perform all FFT passes with triple nested DO loop
;
         do      #WindowLenLog,_end_pass
         move    #FFTbuff,r0     ;initialize A input pointer
         move    r0,r4           ;initialize A output pointer
         lua     (r0)+n0,r1      ;initialize B input pointer
         move    #FFTcoef,r6     ;initialize C input pointer
         lua     (r1)-,r5        ;initialize B output pointer
         move    n0,n1           ;initialize pointer offsets
         move    n0,n4
         move    n0,n5

         do      n2,_end_grp
         move    x:(r1),x1  y:(r6),y0        ;lookup -sine and 
                                             ; -cosine values
         move    x:(r5),a   y:(r0),b         ;preload data
         move    x:(r6)+n6,x0                ;update C pointer

         do      n0,_end_bfy
         mac     x1,y0,b    y:(r1)+,y1       ;Radix 2 DIT
                                             ;butterfly kernel
         macr    -x0,y1,b   a,x:(r5)+    y:(r0),a
         subl    b,a        x:(r0),b     b,y:(r4)
         mac     -x1,x0,b   x:(r0)+,a  a,y:(r5)
         macr    -y1,y0,b   x:(r1),x1
         subl    b,a        b,x:(r4)+  y:(r0),b
_end_bfy
         move    a,x:(r5)+n5    y:(r1)+n1,y1   ;update A and B pointers
         move    x:(r0)+n0,x1   y:(r4)+n4,y1
_end_grp
         move    n0,b1
         lsr     b   n2,a1     ;divide butterflies per group by two
         lsl     a   b1,n0     ;multiply groups per pass by two
         move    a1,n2
_end_pass
        rts

ProcessLoop     ;main loop to process CODEC's samples

RxProcess
        move X:<WindowInpTapPtr,r1 ;r1 to address the input window tap
        move #<WindowLen-1,m1
        move X:<RxTimeToSlide,a   ;a = how many samples are left before
        tst a a,x0                ;if none then do the slide
        jeq <RxSlide              ;immediately
        asl a X:<RxCodecPtr,r2    ;we should make the new window
        add x0,a #<4,n2           ;multiply by DecimateRatio = 3 (hardwired !)
        move a,y1                 ;save the block length in y1, setup n2
        jsr <WaitSampleBlock      ;wait for that many samples to come
        move #BufLen*4-1,m2       ;set m2 so r2 wraps correctly
        move #<AliasFilterLen-1,m4
      if RightChannel
        move (r2)+
      endif
      if RxAGC
        move X:<RxAudioPeak,a   ;update the peak value of the input signal
        move r2,y0              ;save r2
        .loop y1                ;look through the new samples
          move X:(r2)+n2,x0     ;and find the largest one
          cmpm x0,a
          tlt x0,a
        .endl
        abs a y0,r2              ;restore r2
        move a,X:<RxAudioPeak
      endif
        .loop X:<RxTimeToSlide  ;filter, decimate and fill the window's tap
          move #4*(AliasFilterLen-DecimateRatio),n2
          move #AliasFilterInpI,r4 ;r4 to address the input filter shape
          move (r2)-n2            ;move back in the CODEC input buffer
          move #<4,n2             ;n2=4 for easier input adressing
          move r2,x1              ;save r2 (where we start the filter tap)
          clr a X:(r2)+n2,x0 Y:(r4)+,y0
          .loop #AliasFilterLen-1
            mac x0,y0,a X:(r2)+n2,x0  Y:(r4)+,y0
          .endl
          macr x0,y0,a #AliasFilterInpQ,r4
          move x1,r2      ;restore the tap start
          nop
          clr b X:(r2)+n2,x0 Y:(r4)+,y0
          .loop #AliasFilterLen-1
            mac -x0,y0,b X:(r2)+n2,x0  Y:(r4)+,y0
          .endl
          macr -x0,y0,b a,X:(r1) ;a,b = decimated sample (I/Q), restore r2
          move b,Y:(r1)          ;put new sample into the tap
          move X:<RxCarPhase,x0  ;get the receiver NCO's carrier phase
	  move X:<RxCarFreq,a    ;advance the phase
	  add x0,a
	  move a1,X:<RxCarPhase
	  jsr <IQ               ;compute I and Q (modifies a,b,x,y,r0,m0,n0)
          move b,y0
          move X:(r1),x1 a,y1
          mpy x1,y1,a Y:(r1),x0 ;mix the input samples with the NCO
          macr -x0,y0,a
          mpy x1,y0,b
          macr x0,y1,b a,X:(r1)
          move b,Y:(r1)+
        if SPY_RxIQ
          jsr <SpySync
          jcs <SpyDone
          jsr <SpyA
          tfr b,a
          jsr <SpyA
SpyDone   nop
        endif
        .endl
      if RightChannel
        move (r2)-
      endif
        move r2,X:<RxCodecPtr     ;save the CODEC buffer pointer

RxSlide         ;apply window, copy data to FFTbuff
        move X:<RxWindow,r5
        move #<WindowLen-1,m5
        move #FFTbuff,r0
        move #<WindowLen-1,m0
        .loop #WindowLen                ;apply input window
          move X:(r1),x0 Y:(r5)+,y0     ;x0 = sample_I, y0 = input window
          mpyr x0,y0,a Y:(r1)+,x0       ;a = x0*y0, x0 = sample_Q
          mpyr x0,y0,a a,X:(r0)         ;a = x0*y0, save windowed I
          move a,Y:(r0)+                ;save windowed Q into FFTbuff
        .endl
        move r1,X:<WindowInpTapPtr

        if SPY_RxWindow
         jsr <SpySync
         .if <cc>
          .loop #WindowLen
            move X:(r1)+,a
            jsr <SpyA
            nop
          .endl
          .loop #WindowLen
            move Y:(r1)+,a
            jsr <SpyA
            nop
          .endl
         .endi
        endif

        jsr <DoFFT              ;execute FFT

      if RxAGC                  ;update the signals power (mean square)
        move #FFTbuff,r0
        move #<0,m0
        move #WindowLen/2,n0
        clr a X:(r0),x0
        .loop #WindowLen-1
          mac x0,x0,a Y:(r0)+n0,x0
          mac x0,x0,a X:(r0),x0
        .endl
        mac x0,x0,a Y:(r0)+n0,x0
        macr x0,x0,a X:<RxAudioMS,x1
        move a,x0 #RxAGC_MSfollow,y0
        mpy x0,y0,a #1.0-RxAGC_MSfollow,y0
        macr x1,y0,a
        move a,X:<RxAudioMS

       if SPY_PeakMS
        clr b X:<RxAudioPeak,a
      if !RxAGCsensePeak
        move b,X:<RxAudioPeak
      endif
        jsr <SpyA
        move X:<RxAudioMS,a
        jsr <SpyA
       endif

      if MonitorRxAGC
        UpLED clr
        DownLED clr
      endif

      if RxAGCavoidDCD
       jset #2,X:<RxState,CheckMS_OK
      endif

      move X:<RxCodecPtr,r2     ;set r2 to address the CODEC's buffer
      move #BufLen*4-1,m2       ;set m2 so r2 wraps in the CODEC's buffer
                                ;we will need r2/m2 for RxGainUp/Down
      if RxAGCsensePeak
        move #RxAGCmaxPeak,x0   ;check if peak is above the given max.
        move X:<RxAudioPeak,b
        cmp x0,b #<0,b
        move b,X:<RxAudioPeak   ;clear the peak
        jcc <GainDown           ;reduce gain if the peak is too high
      endif

        move #RxAGCminMS,x0     ;see, if the MS is below the minimal level
        cmp x0,a
        jcc <CheckMSmax         ;if not, then check if above the max. level
                                ;if so, increase the CODEC's input gain
        move X:<RxAGCcount,a    ;decrement the timeout
        move #>1,x0
        sub x0,a 
        move a,X:<RxAGCcount
        jgt <CheckMS_OK         ;leave if not yet zero
        jsr <RxGainUp           ;increase the CODEC's gain
        jcs <CheckMS_OK
        move #0.7071,y0         ;increase the MS to follow
        move X:<RxAudioMS,x0    ;the gain change faster
        mpyr x0,y0,a #>RxAGCfall,x0 ;preset the AGC "timer" to prohibit
        asl a x0,X:<RxAGCcount      ;gain increase for the RxAGCfall period
        move a,X:<RxAudioMS
      if MonitorRxAGC
        UpLED set
      endif
        jmp <CheckMS_OK

CheckMSmax                      ;is the MS above the given maximum
        move #>RxAGChold,y0     ;preset the AGC "timer" to prohibit gain increase
        move #RxAGCmaxMS,x0     ;for the RxAGChold period
        cmp x0,a y0,X:<RxAGCcount ;MS above maximum ?
        jcs <CheckMS_OK         ;if not then we are done
GainDown
        jsr <RxGainDown         ;otherwise decrease the CODEC's gain
        jcs <CheckMS_OK
        move #0.7071,y0         ;decrease the MS to follow expected
        move X:<RxAudioMS,x0    ;gain reduction faster
        mpyr x0,y0,a #>RxAGChold,x0     ;initialize the AGC hold count-down
        move a,X:<RxAudioMS
        move x0,X:<RxAGCcount
      if MonitorRxAGC
        DownLED set
      endif

CheckMS_OK

      endif

        if SPY_RxFFT
         jsr <SpySync
         .if <cc>
          move #FFTbuff,r0
          move #<0,m0
          move #WindowLen/2,n0
          .loop #WindowLen
            move X:(r0),a
            jsr <SpyA
            move Y:(r0)+n0,a
            jsr <SpyA
            nop
          .endl
         .endi
        endif

        move X:<RxPhaseCorr,r5         ;adjust FFT phases,
        move #<WindowLen-1,m5          ;and copy the selected FFT bins
        move X:<RxTimeToSlide,n5       ;into the decoder's buffer
        move X:<RxPipePtr,r4
        move #<DataCarriers*RxPipeLen-1,m4
        move #<DataCarriers,n4         ;this code is a bit tricky because we want
        move (r5)+n5                   ;to pick-up only selected bins
        move r5,X:<RxPhaseCorr         ;out of a bit-reversly addresses buffer
        move r5,n5
        move #FFTcoef+WindowLen/2,r5
        rep #FirstDataCarr
          move (r5)-n5
        move #FFTbuff,r0
        move #<@rvb(FirstDataCarr,WindowLenLog),n0
        move #<0,m0
        move (r4)+n4
        move r4,X:<RxPipePtr
        move (r0)+n0                    ;move ahead by FirstDataCarrier
        move #<@rvb(DataCarrSepar,WindowLenLog),n0
        .loop #DataCarriers
          move L:(r5)-n5,y      ;get the phase correction
          move L:(r0)+n0,x      ;get the FFT output
          mpy x1,y1,a           ;turn the FFT output by the given correction
          macr -x0,y0,a (r5)-n5 ;(r5)-n5 is done 3 times to jump
          mpy x1,y0,b (r5)-n5   ;by 3 (DataCarrSepar) bins
          macr x0,y1,b a,X:(r4)
          move b,Y:(r4)+
        .endl

        if SPY_RxCarIQ
         jsr <SpySync
         .if <cc>
          move X:<RxPipePtr,r4
          move #<DataCarriers*RxPipeLen-1,m4
          .loop #DataCarriers
            move X:(r4),a
            jsr <SpyA
            move Y:(r4)+,a
            jsr <SpyA
            nop
          .endl
         .endi
        endif

        if SPY_RxCarFreq
          move X:<RxCarFreq,a
          jsr <SpyA
        endif
        if SPY_RxStateRou
          move X:<RxStateRoutine,a
          rep #8
            asl a
          jsr <SpyA
        endif

      if RxCtrlUpDown
        move X:<RxStateRoutine,r0
        move X:<RxUpDownTime,a
        tst a #>1,x0
        jle (r0)
        sub x0,a
        move a,X:<RxUpDownTime
        jlt (r0)
        PushUp clr
        PushDown clr
        jmp (r0)
      else
        move X:<RxStateRoutine,r0
        nop
        jmp (r0)
      endif

RxStartIdle                     ;carrier lost or Rx initialized
        bclr #2,X:<RxState      ;declare "free channel"
        if !SPY
         if !TxPTThack
          caroff                ;tell LEONID the carrier is off
         endif
        endif
        RedLED clr              ;turn off the red LED

        move #>RxDoIdle,x0      ;set Rx handler to "idle"
        move x0,X:<RxStateRoutine
        move #>ToneWindowInp,x0 ;set FFT window to "tone"
        move x0,X:<RxWindow
        move #>SymbolLen/2,x0   ;make slides every half-symbol
        clr a x0,X:<RxTimeToSlide
;        move #30.0*DecimateRatio*2.0/SampleFreq,a ;*** DEBUG ***
        move a,X:<RxCarFreq     ;zero the tuning corrector
        move #-1.0,a            ;set the acceptance counter to -1.0
        move a,X:<RxAcceptCount ;
        clr a #<RxTuneCorrel,r0 ;clear the correlation accumulators
        move #<TuneCarriers*8-1,m0
        rep #TuneCarriers*8
          move a,L:(r0)+
        jmp <RxDone

RxDoIdle                        ;hunt for the tuning tones
        move X:<RxPipePtr,r0
        move #<DataCarriers*RxPipeLen-1,m0
        move #<(FirstTuneCarr-FirstDataCarr)/DataCarrSepar,n0
        move #<DataCarriers*RxPipeLen-1,m1
        move (r0)+n0
        move #<DataCarriers,n0
        move #<TuneCarrSepar/DataCarrSepar,n1
        lua (r0)-n0,r1
        move #<TuneCarrSepar/DataCarrSepar,n0
        move #<RxTuneCorrel,r4
        move #<TuneCarriers*8-1,m4
        move #<8-2,n4
        .loop #TuneCarriers
          move L:(r0),x        ;compute carrier's power
          mpy x0,x0,a x1,x0
          mac x0,x0,a L:(r4),b ;average the power
          LongDecayAver a,b,#RxAverFollow
          move L:(r1)+n1,x     ;correlate this I/Q with the previous one
          move L:(r0)+n0,y
          mpy x1,y1,a b,L:(r4)+ ;save the averaged power
          mac x0,y0,a L:(r4),b ;average the I/Q correlations
          LongDecayAver a,b,#RxAverFollow
          mpy -x0,y1,a b,L:(r4)+
          mac x1,y0,a L:(r4),b
          LongDecayAver a,b,#RxAverFollow
          move b,L:(r4)+n4
        .endl

        move #>1,x1            ;and count them
        move #<0,x0
        move #<8,n4
        .loop #TuneCarriers
          move (r4)+
          move L:(r4)+,a      ;estimate the correlation amplitude
          abs a L:(r4)-,b     ;take I and Q parts, compute absolute
          abs b               ;values
          cmp b,a             ;which is greater ?
          .if <lt>            ;if I less than Q
            tfr b,a L:(r4),b  ;swap I with Q
            abs b
          .endi               ;a = greater part, b = smaller part
          asr b               ;compute 2*(a+b/4+b/16)
          asr b
          add b,a
          asr b (r4)-
          addl b,a L:(r4)+n4,b ;load carrier's power
          cmp a,b x0,a        ;and compare: a = 2*correl. power, b = sig. power
          .if <lt>            ;if power < 2*correlation
            add x1,a          ;count "good" carriers
          .endi
          move a,x0           ;save the counter
        .endl

        if SPY_RxCorrel
          move #<8,n4
          jsr <SpySync
          .if <cc>
            .loop #8
              move L:(r4)+,a
              rep #8
                asl a
              jsr <SpyA
              nop
            .endl
            move (r4)-n4
          .endi
        endif

                               ;see which carriers show high correlation
        tfr x0,a #>TuneCarriers-1,x1    ;if more than 3 carriers are "good"
        cmp x1,a #1.0/@cvf(2*RxMinTune),x0
        move X:<RxAcceptCount,b
        .if <ge>                        ;then increment the acceptance
          if MonitorRxState
            UpLED set
          endif
          add x0,b                      ;counter
        .else                           ;otherwise decrement it
          if MonitorRxState
            UpLED clr
          endif
          sub x0,b
        .endi
        move b,X:<RxAcceptCount
      if RxEarlyDCD
        cmp x1,a
        .if <ge>
          bset #2,X:<RxState      ;declare "busy channel"
          if !SPY
           if !TxPTThack
            caron                 ;tell LEONID the carrier is on
           endif
          endif
          RedLED clr              ;turn on the red LED
        .else
          bclr #2,X:<RxState      ;declare "free channel"
          if !SPY
           if !TxPTThack
            caroff                ;tell LEONID the carrier is off
           endif
          endif
          RedLED clr              ;turn off the red LED
        .endi
        move X:<RxAcceptCount,b
        tst b
      endif
        jmi <RxDone     ;if negative, go on with present state
                        ;but if counter above zero => "tune detect" condition
          move #<RxTunePhase,r0         ;compute the correlation phases
          move #<TuneCarriers-1,m0      ;they will tell us by how much
          move #<8-2,n4                 ;are me off in frequency
          .loop #TuneCarriers
            move (r4)+
            move L:(r4)+,a              ;a = correlation.I
            move L:(r4)+n4,b            ;b = correlation.Q
            jsr <Phase48                ;compute the phase
            move a,Y:(r0)+              ;save it for later
          .endl
          move Y:(r0)+,a        ;find the max. and min. phase (to reject them later)
          move a,b              ;initialize min and max with the first phase
          .loop #TuneCarriers-1 ;seek min and max
            move Y:(r0)+,x0     ;get next phase
            cmp x0,a            ;higher than the maximum ?
            tlt x0,a            ;if so then substitute the maximum
            cmp x0,b            ;lower than the minimum ?
            tgt x0,b            ;if so then substitute the minimum
          .endl
          add b,a Y:(r0)+,x0    ;sum up all four deviations
          rep #TuneCarriers-1   ;but subtract the max.and min.
            sub x0,a Y:(r0)+,x0 ;infact we compute "-sum"...
          sub x0,a X:<RxCarFreq,b
          asr a #4.0/WindowLen,x0 ;take the average (works only for TuneCarriers=4)
          move a,x1             ;load the dev->freq. factor
          macr x0,x1,b          ;correct the tuning freq.
          move b,X:<RxCarFreq
          if MonitorRxState
            UpLED clr
          endif

       if SPY_RxToneTune
         jsr <SpySync
         .if <cc>
          .loop #8*TuneCarriers
            move L:(r4)+,a
            rep #8
              asl a
            jsr <SpyA
            nop
          .endl
          .loop #TuneCarriers
            move Y:(r0)+,a
            jsr <SpyA
            nop
          .endl
          tfr x0,a
          jsr <SpyA
          tfr x1,a
          jsr <SpyA
          tfr b,a
          jsr <SpyA
          clr a
          .loop #64-8*TuneCarriers-TuneCarriers-3
            jsr <SpyA
            nop
          .endl
         .endi
       endif

          jmp <RxStartTune      ;switch to the data state

RxStartTune                     ;tune in preamble detected
        bset #2,X:<RxState      ;declare "channel busy"
        if !SPY
          if !TxPTThack
            caron       ;modifies r0/m0 !
          endif
        endif
        RedLED set            ;turn on the red LED (DCD indicator)

        move #>RxDoTune,x0      ;set Rx handler to "tune"
        move x0,X:<RxStateRoutine
        move #>DataWindowInp,x0 ;set FFT window to "data"
        move x0,X:<RxWindow
        move #>2*RxTuneTimeout,x0       ;set timeout on the "tune" phase
        move x0,X:<RxStateCounter
        move #-1.0,a            ;set the acceptance counter to -1.0
        clr a a,X:<RxAcceptCount ;
;       move #15.0*DecimateRatio*2.0/SampleFreq,a ;*** DEBUG ***
;       move a,X:<RxCarFreq
        move #<RxTuneCorrel,r4   ;initialize the correl. accumulators
        move #<TuneCarriers*8-1,m4
        .loop #TuneCarriers
          clr b L:(r4)+,a
          rep #2
            move a,L:(r4)+
          move b,L:(r4)+
          rep #3
            move a,L:(r4)+
          move b,L:(r4)+
        .endl
        jmp <RxDone

RxDoTune                        ;tune-in and wait for the sync. preamble
        move X:<RxStateCounter,a        ;counter has expired ?
        move #>1,x0
        sub x0,a
        jmi <RxStartIdle                ;go to the idle phase if it did
        move a,X:<RxStateCounter

        move X:<RxPipePtr,r2
        move #<DataCarriers*RxPipeLen-1,m2
        move #<(FirstTuneCarr-FirstDataCarr)/DataCarrSepar,n2
        move #<DataCarriers*RxPipeLen-1,m0
        move (r2)+n2
        move #<DataCarriers,n2
        move r2,r0                      ;(r0) = most recent I/Q
        move #<TuneCarrSepar/DataCarrSepar,n0
        move #<DataCarriers*RxPipeLen-1,m1
        move (r2)-n2
        move r2,r1                      ;(r1) = most recent - 1
        move #<TuneCarrSepar/DataCarrSepar,n1
        move (r2)-n2                    ;(r2) = most recent - 2
        move #<TuneCarrSepar/DataCarrSepar,n2
        bchg #0,X:<RxState              ;test and swap the
        .if <cs>                        ;at-symbol flag
          move #<RxTuneCorrel,r4
        .else
          move #<RxTuneCorrel+4,r4
        .endi
        move #<TuneCarriers*8-1,m4
        move #<8-3,n4
        .loop #TuneCarriers
          move L:(r0),x        ;compute carrier's power
          mpy x0,x0,a x1,x0
          mac x0,x0,a L:(r4),b ;average the power
          LongDecayAver a,b,#RxAverFollow-1
          move L:(r0)+n0,x      ;correlate with current - 1
          move L:(r1)+n1,y
          mpy x1,y1,a b,L:(r4)+ ;only the I-type correlation
          mac x0,y0,a L:(r4),b
          LongDecayAver a,b,#RxAverFollow-1
          move L:(r2)+n2,y      ;correlate with current - 2
          mpy x1,y1,a b,L:(r4)+
          mac x0,y0,a L:(r4),b
          LongDecayAver a,b,#RxAverFollow-1
          mpy -x0,y1,a b,L:(r4)+
          mac x1,y0,a L:(r4),b
          LongDecayAver a,b,#RxAverFollow-1
          move b,L:(r4)+n4
        .endl

        jset #0,X:<RxState,RxDone       ;do the rest only
                                        ;when we are at-symbol

        if SPY_RxCorrel
          move #<8,n4
          jsr <SpySync
          .if <cc>
            .loop #8
              move L:(r4)+,a
              rep #8
                asl a
              jsr <SpyA
              nop
            .endl
            move (r4)-n4
          .endi
        endif

                               ;see which carriers pass the criteria
        move #>1,x1            ;and count them
        move #<0,x0
        move #<4,n4
        move m4,m5
        move r4,r5
        move #<2,n5
        .loop #TuneCarriers
          move L:(r4)+n4,a      ;load the carrier's power from at-symbol
          move L:(r4),b         ;and from inter-symbol
          cmp b,a               ;see which is greater
          tlt b,a r4,r5         ;take the greater one and remember in r5
          asr a (r4)+n4         ;which data were taken
          asr a L:(r5+n5),b     ;load the I-correl. -2
          neg b                 ;see if (-correl) > power/4
          cmp a,b x0,a          ;or is it safer to take power/2 ?
          .if <gt>              ;count good carriers
            add x1,a
            move a,x0
          .endi
          move r4,r5
        .endl

        tfr x0,a #>TuneCarriers-1,x1    ;if more than 3 carriers are "good"
        cmp x1,a #1.0/@cvf(RxMinSync),x0
        move X:<RxAcceptCount,a
        .if <ge>                        ;then increment the acceptance
          if MonitorRxState
            DownLED set
          endif
          add x0,a                      ;counter
        .else                           ;otherwise decrement it
          if MonitorRxState
            DownLED clr
          endif
          sub x0,a
        .endi
        move a,X:<RxAcceptCount
        jmi <RxDone             ;if counter negative, the time to switch
                                ;has not yet come
                                ;now we find the symbol-timing
                                ;and mis-tune for every carrier
        move #<RxTunePhase,r0
        move #<TuneCarriers-1,m0
        move #<RxSyncDelay,r1
        move #<TuneCarriers-1,m1
        move #<4,n4
        move m4,m5
        move r4,r5
        move #<2,n5
        .loop #TuneCarriers
          move L:(r4)+n4,a      ;load the carrier's power from at-symbol
          move L:(r4),b         ;and from inter-symbol
          cmp b,a #>SymbolLen/2,x1 ;see which is greater
          .if <lt>              ;take the greater one and remember in r5
            tfr b,a r4,r5       ;which data were taken
            move #>SymbolLen,x1 ;if the "inter-symbol" stronger
          .endi                 ;we have to shift the timing by half-symbol
          move x1,Y:(r1)        ;save the approx. timing
          move L:(r5+n5),b      ;compute ( [0].I - [-2].I ) / 2
          sub b,a (r5)+
          asr a L:(r5)+,b       ;b = [-1].I
          jsr <Phase48
          move a,x1 Y:(r1),a
          move #>SymbolLen,x0
          macr -x0,x1,a (r4)+n4
          move a,Y:(r1)+
          move L:(r5)+,a        ;load [-2].I
          neg a L:(r5)+,b       ;and [-2].Q
                                ;invert this vector because of the bi-phase modulation
          jsr <Phase48          ;compute the phase
          move a,Y:(r0)+        ;save for later tuning correction
          move r4,r5
        .endl

                              ;correct symbol timing
        move Y:(r1)+,a        ;find the max. and min. phase (to reject them later)
        move a,b              ;initialize min and max with the first phase
        .loop #TuneCarriers-1 ;seek min and max
          move Y:(r1)+,x0     ;get next phase
          cmp x0,a            ;higher than the maximum ?
          tlt x0,a            ;if so then substitute the maximum
          cmp x0,b            ;lower than the minimum ?
          tgt x0,b            ;if so then substitute the minimum
        .endl
        add b,a Y:(r1)+,x0    ;sum up all four deviations
        neg a
        rep #TuneCarriers-1   ;but subtract the max.and min.
          add x0,a Y:(r1)+,x0
        add x0,a
        asr a                 ;take the average (works only for TuneCarrier=4)
        move a,X:<RxTimeToSlide

                              ;correct the frequency error
        move Y:(r0)+,a        ;find the max. and min. phase (to reject them later)
        move a,b              ;initialize min and max with the first phase
        .loop #TuneCarriers-1 ;seek min and max
          move Y:(r0)+,x0     ;get next phase
          cmp x0,a            ;higher than the maximum ?
          tlt x0,a            ;if so then substitute the maximum
          cmp x0,b            ;lower than the minimum ?
          tgt x0,b            ;if so then substitute the minimum
        .endl
        add b,a Y:(r0)+,x0    ;sum up all four deviations
        rep #TuneCarriers-1   ;but subtract the max.and min.
          sub x0,a Y:(r0)+,x0 ;infact we compute "-sum"...
        sub x0,a X:<RxCarFreq,b
        asr a #2.0/WindowLen,x0 ;take the average (works only for TuneCarrier=4)
        move a,x1             ;load the dev->freq. factor
        macr x0,x1,b          ;correct the carrier freq.
        move b,X:<RxCarFreq

        if MonitorRxState
          DownLED clr
        endif

       if SPY_RxSyncTune
         jsr <SpySync
         .if <cc>
          .loop #8*TuneCarriers ;send correlation data
            move L:(r4)+,a
            rep #8
              asl a
            jsr <SpyA
            nop
          .endl
          .loop #TuneCarriers   ;send timing data
            move Y:(r1)+,a
            rep #8
              asl a
            jsr <SpyA
            nop
          .endl
          .loop #TuneCarriers   ;send freq. error data
            move Y:(r0)+,a
            jsr <SpyA
            nop
          .endl
          move X:<RxTimeToSlide,a
          rep #8
            asl a
          jsr <SpyA
          move X:<RxCarFreq,a
          jsr <SpyA
          clr a
          .loop #64-8*TuneCarriers-2*TuneCarriers-2
            jsr <SpyA
            nop
          .endl
         .endi
       endif

        jmp <RxStartData

RxStartData
        move #>RxDoData,x0      ;set Rx handler to "tune"
        move x0,X:<RxStateRoutine
        move #>RxUpdateHold,x0     ;set time when the statistics update
        move x0,X:<RxStateCounter  ;should be suspended
;       move #5.0*DecimateRatio*2.0/SampleFreq,a ;*** DEBUG ***
;       move a,X:<RxCarFreq
        move #0.5,a                ;preset the acceptance counter
        move a,X:<RxAcceptCount
        bclr #1,X:<RxState        ;declare "update not allowed"
        clr a #RxDataCorrel,r4    ;clear the correlation accumulators
        move #<DataCarriers*3-1,m4
        rep #DataCarriers*3
          move a,L:(r4)+
    if 0
        move #<RxTuneCorrel,r4   ;copy the carrier powers from Tune to Data
        move #<4,n4
        move #<TuneCarriers*8-1,m4
        move #RxDataCorrel,r5    ;init. the DCD/tune/sync. correlations
        move #<4*DataCarriers-1,m5
        .loop #TuneCarriers     ;this is a real bad piece of code...
          move L:(r4)+n4,a
          move L:(r4)+n4,b
          add b,a
          asr a
          asr a
          .loop #(DataCarriers+1)/TuneCarriers
            clr b a,L:(r5)+
            rep #3
              move b,L:(r5)+
            nop
          .endl
          nop
        .endl
    endif
        jmp <RxDone

RxDoData                        ;data phase
                                ;receive data symbols, tune-in & sync. adjust
                                ;monitor DCD condition, when it drops
                                ;change to Idle state.
        move #>SymbolLen/2,x0   ;reset the slide interval
        move x0,X:<RxTimeToSlide
        bchg #0,X:<RxState      ;test and flip the at-symbol flag
        jcc <RxDone             ;we have not much to do for the "inter-symbol"

        if SPY_RxDataIQ
         jsr <SpySync
         .if <cc>
          move X:<RxPipePtr,r4
          move #<DataCarriers*RxPipeLen-1,m4
          move #<DataCarriers,n4
          nop
          move (r4)-n4
          .loop #2*DataCarriers
            move X:(r4),a
            jsr <SpyA
            move Y:(r4)+,a
            jsr <SpyA
            nop
          .endl
         .endi
        endif

        move X:<RxPipePtr,r2
        move #<DataCarriers*RxPipeLen-1,m2
        move #<DataCarriers,n2
        move #<DataCarriers*RxPipeLen-1,m0
        move r2,r0                      ;(r0) = most recent I/Q
        move (r2)-n2
        move (r2)-n2                    ;(r2) = most recent - 2
        clr a #RxDataCorrel,r4
        move #<3*DataCarriers-1,m4
        clr b #<2,n4
        .loop #DataCarriers
                             ;shift the bit word
          lsr a1  L:(r2),x   ;get I/Q [-2]
          lsr b1  L:(r0),y   ;get I/Q [0]
          mpy x1,y1,a a1,X:<RxWord ;scalar multiply
          mac x0,y0,a b1,Y:<RxWord
          mpy -x0,y1,b        ;scalar multiply but with "[-2] I/Q"
          mac x1,y0,b         ;turned by +90 degrees
          cmpm b,a            ;which multiply result is greater ?
          jgt <Rx00or11       ;jump if |a| > |b| (di-bit = 00 or 11)
          tst b               ;b positive => 10, negative => 01
          jpl <Rx10
Rx01        bset #DataCarriers-1,X:<RxWord
            neg b a,L:<RxTmpQ
            tfr b,a b,L:<RxTmpI
            jmp <RxCarLoop_cont

Rx10      bset #DataCarriers-1,Y:<RxWord
          neg a b,L:<RxTmpI
          tfr b,a a,L:<RxTmpQ
          jmp <RxCarLoop_cont

Rx00or11  tst a               ;a positive => 11, negative => 00
          jpl <Rx11
Rx00        neg a
            neg b a,L:<RxTmpI
            move b,L:<RxTmpQ

            jclr #1,X:<RxState,RxCarLoop_cont ;avoid the update if not yet allowed
                              ;if 00 (180 phase swap) do symbol timing corr.
            move y1,a         ;subtract: following - preceeding
            sub x1,a  y0,b
            sub x0,b  a,y1
            move b,y0
            move L:(r2+n2),x    ;get center I,Q
            mpy x0,y0,a (r4)+   ;scalar multiply
            mac x1,y1,a L:(r4),b  ;a = inter-bit direction for timing
            LongDecayAver a,b,#RxDataSyncFollow
            move b,L:(r4)-
            move L:<RxTmpI,a
            jmp <RxCarLoop_cont

Rx11         bset #DataCarriers-1,X:<RxWord
             bset #DataCarriers-1,Y:<RxWord
             move a,L:<RxTmpI
             move b,L:<RxTmpQ
;            jmp <RxCarLoop_cont

RxCarLoop_cont                  ;a = at-bit amplitude
          jclr #1,X:<RxState,RxCarLoop_end      ;avoid the update if not yet allowed
          move L:(r4),b         ;we treat this as the carrier power
          LongDecayAver a,b,#RxDataSyncFollow
          move b,L:(r4)+n4
        if SPY_RxCrossedPhaseAver
          move X:(r4),a
          jsr <SpyA
        endif
          move L:<RxTmpI,a
          move L:<RxTmpQ,b
          jsr <Phase48
        if SPY_RxCrossedPhase
          jsr <SpyA
        endif
          move a,x0 #1.0-DCDTuneAverWeight,y0
          mpyr x0,x0,a X:(r4),x1
          mpy x1,y0,a a,x1 #DCDTuneAverWeight,y1
          macr x1,y1,a Y:(r4),x1
          mpy x1,y0,a a,X:(r4)
          macr x0,y1,a
          move a,Y:(r4)+
RxCarLoop_end
        if SPY_RxCrossedIQ
         jsr <SpySync
         .if <cc>
           move L:<RxTmpI,a
           rep #8
             asl a
           jsr <SpyA
           move L:<RxTmpQ,a
           rep #8
             asl a
           jsr <SpyA
         .endi
        endif
          move (r0)+
          clr a (r2)+
          move X:<RxWord,a1
          move Y:<RxWord,b1
        .endl

        if Interleave
          move X:<RxWord,a1
          jsr <RxDeInlv
          move a1,X:<RxWord
          move Y:<RxWord,a1
          jsr <RxDeInlv
          move a1,Y:<RxWord
        endif

        if FEC==1
          move X:<RxWord,a1
          jsr <FEC1511Decode
          move a1,X:<RxWord
          move b1,X:<RxErrPattern
          move Y:<RxWord,a1
          jsr <FEC1511Decode
          move a1,Y:<RxWord
          move X:<RxErrPattern,x0
          or x0,b
          move b1,X:<RxErrPattern
         if FECflickerRedLED
          .if <ne>
            RedLED clr
          .else
            RedLED set
          .endi
         endif
        endif

        if FEC==3
          move X:<RxWord,a1
          jsr <WalshDecode
          move a1,X:<RxWord
          move b1,X:<RxErrPattern
          move Y:<RxWord,a1
          jsr <WalshDecode
          move a1,Y:<RxWord
          move X:<RxErrPattern,x0
          or x0,b
          move b1,X:<RxErrPattern
         if FECflickerRedLED
          .if <ne>
            RedLED clr
          .else
            RedLED set
          .endi
         endif
        endif

        if SPY_RxDataWord
          move X:<RxWord,a
          rep #8
            asl a
          jsr <SpyA
          move Y:<RxWord,a
          rep #8
            asl a
          jsr <SpyA
        endif

        if !SPY
          move X:<RxWord,a1
          jsr <PutBitBatch
          move Y:<RxWord,a1
          jsr <PutBitBatch
        endif

        jclr #1,X:<RxState,RxUpdate_wait

        if SPY_DCDaver
          clr a (r4)+n4
          .loop #DataCarriers
            move X:(r4)+,x0
            add x0,a (r4)+n4
          .endl
          move (r4)-n4
;          rep #4
;            asr a
          jsr <SpyA
        endif

        if SPY_DataTuneAver
          clr a (r4)+n4
          .loop #DataCarriers
            move Y:(r4)+,x0
            add x0,a (r4)+n4
          .endl
          move (r4)-n4
          rep #4
            asr a
          jsr <SpyA
        endif

        if SPY_DataSyncAver
          clr a
          .loop #DataCarriers
            move L:(r4)+,b
            add b,a (r4)+n4
          .endl
          rep #4
            asl a
          jsr <SpyA
          clr a (r4)+
          .loop #DataCarriers
            move L:(r4)+,b
            add b,a (r4)+n4
          .endl
          move (r4)-
          rep #4
            asl a
          jsr <SpyA
        endif


        move #DCDThreshold,y0
        clr b #>1,y1            ;init. sum and increment
        move (r4)+n4
        .loop #DataCarriers     ;count the carriers which has the average
          move X:(r4)+,a        ;phase deviation below the threshold
          cmp y0,a
          .if <lt>
            add y1,b
          .endi
        if SPY_RxPhaseAbsAver
          jsr <SpyA
        endif
          move (r4)+n4
        .endl
        move (r4)-n4

        if SPY_DCDnum
          tfr b,a
          rep #8
            asl a
          jsr <SpyA
        endif

        move #>DataCarriers/2,x0        ;if more than half carriers
        cmp x0,b #1.0/@cvf(DCDMaxDrop),x1 ;are OK
        move X:<RxAcceptCount,a
        .if <ge>                        ;then increment the acceptance
          if MonitorRxState
            YellowLED set
          endif
          add x1,a                      ;counter
        .else                           ;otherwise decrement it
          if MonitorRxState
            YellowLED clr
          endif
          sub x1,a
        .endi
        cmp x0,b a,X:<RxAcceptCount
        .if <ge>                ;bring back the carrier's decision
          move (r4)+n4          ;if positive then correct the frequency error
          move Y:(r4)+,a        ;find the max. and min. phase (to reject them later)
          tfr a,b (r4)+n4       ;initialize min and max with the first phase
          .loop #DataCarriers-1 ;seek min and max
            move Y:(r4)+,x0     ;get next phase
            cmp x0,a (r4)+n4    ;higher than the maximum ?
            tlt x0,a            ;if so then substitute the maximum
            cmp x0,b            ;lower than the minimum ?
            tgt x0,b            ;if so then substitute the minimum
          .endl
          add b,a Y:(r4)+,x0    ;sum up all deviations
          move (r4)+n4
          .loop #DataCarriers-1 ;but subtract the max. and min.
            sub x0,a Y:(r4)+,x0 ;infact we compute "-sum"...
            move (r4)+n4
          .endl
          sub x0,a X:<RxCarFreq,b
          asr a (r4)-n4
          asr a #2.0/WindowLen*RxFreqFollowWeight/4,x0
          move a,x1             ;load the dev->freq. factor
          macr x0,x1,b          ;correct the carrier freq.
          move b,X:<RxCarFreq
        .endi

        jclr #23,X:<RxAcceptCount,RxDone ;if counter positive we go on
                                         ;with the data phase

        if RxCtrlUpDown         ;check if we need to correct the tranceiver's
          move X:<RxCarFreq,a   ;frequency
          move #UpDownCorrThres*DecimateRatio*2.0/SampleFreq,x0
          cmpm x0,a
          jlt <RxStartIdle      ;jump if frequency error lower than threshold
          tst a #>2*UpDownPulseLen,x0
          move x0,X:<RxUpDownTime
        if UpDownReverse
          .if <pl>
        else
          .if <mi>
        endif
            PushUp set
          .else
            PushDown set
          .endi
        endif

        jmp <RxStartIdle        ;if counter negative, the signal
                                ;is assumed to be gone
                                ;and we switch to the idle state

RxUpdate_wait                           ;don't update DCD/sync. statistics
        move X:<RxStateCounter,a        ;but wait until the counter
        move #>1,x0                     ;expires.
        sub x0,a
        move a,X:<RxStateCounter
        jpl <RxDone                     ;if the counter has expired
        bset #1,X:<RxState              ;then set the allow-update flag
        jmp <RxDone

RxDone

TxProcess
        move #>DecimateRatio*SymbolLen,a
        move X:<TxCodecPtr,r2
        jsr <CheckSampleBlock   ;time to transmit a new symbol has come ?
        jcs <ProcessLoop        ;jump if not yet...
        move X:<TxStateRoutine,r0
        move #BufLen*4-1,m2
        jmp (r0)

TxStartIdle
        PTT clr                 ;turn off the PTT
        move #>TxDoIdle,x0
        move x0,X:<TxStateRoutine
        move #>TxMinIdle,x0     ;obligatory silence period
        move x0,X:<TxStateCounter
TxDoIdle                        ;send nothing (silence)
        move X:<TxStateCounter,a ;when counter expires
        tst a #>1,x0            ;check if any data for transmition
      if SPY
        jle <TxStartTune
      else
        jle <TxCheckPTTreq
      endif
        sub x0,a
        move a,X:<TxStateCounter
        jmp <TxSilence

      if !SPY
TxCheckPTTreq
       if TxPTThack
        jclr #0,X:<TxState,TxSilence    ;wait if no PTT-ON request
        jset #2,X:<RxState,TxSilence    ;wait if channel is busy
        jsr <Rand48                     ;Persistance decision
        move #TxPersistance,x0
        cmpm x0,a                       ;if random number below persistance
        jlt <TxStartTune                ;then push the PTT and begin
       else                             ;the tuning sequence
        jset #2,X:<RxState,TxSilence    ;make sure the channel is free
        jset #0,X:<TxState,TxStartTune  ;if the PTT-ON was requested
       endif                            ;then begin the tuning sequence
      endif

TxSilence
        clr a #FFTbuff,r4       ;set all FFT bins to 0
        move #<WindowLen-1,m4   ;as to make silence
        .loop #WindowLen
          move a,L:(r4)+
        .endl
        jmp <TxDoWindow         ;avoid the FTT, zero input => zero output
                                ;jump directly to the output interpolation

TxStartTune
        PTT set                 ;turn on the PTT
        move #>ToneWindowOut,x0 ;set the window to "pure tone"
        move x0,X:<TxWindow
        move #>TxDoTune,x0      ;set the state routine
        move x0,X:<TxStateRoutine
        move #>TxTuneLen,x0     ;initialize the count-down
        move x0,X:<TxStateCounter
        move #TxTuneIniVect,r4
        move #<TuneCarriers-1,m4
        move #TxTuneVect,r5
        move #<TuneCarriers-1,m5
        .loop #TuneCarriers             ;copy initial-phase vectors
           move L:(r4)+,ab              ;into the TxTuneVect buffer
           move ab,L:(r5)+
        .endl
TxDoTune                                ;send tune-up sequence
        move X:<TxStateCounter,a
        tst a #>1,x0
        jle <TxStartSync
        sub x0,a
        move a,X:<TxStateCounter
        clr a #FFTbuff,r4       ;set all FFT bins to 0
        move #<WindowLen-1,m4
        .loop #WindowLen
          move a,L:(r4)+
        .endl
        move #TxTuneVect,r5     ;copy tune vectors into the FFT buffer
        move #<TuneCarriers-1,m5
        move #FFTbuff+FirstTuneCarr,r4
        move #<TuneCarrSepar,n4
        .loop #TuneCarriers
          move L:(r5),ab        ;get I/Q
          jclr #0,r4,TxTune_cont ;flip odd carrier's phase
            neg a                ;to make continous wave on the output
            neg b
TxTune_cont move ab,L:(r5)+     ;save the I/Q
          move ab,L:(r4)+n4     ;put it into the FFT buffer
        .endl
        jmp <TxDoFFT

TxStartSync
        move #>DataWindowOut,x0
        move x0,X:<TxWindow
        move #>TxDoSync,x0
        move x0,X:<TxStateRoutine
        move #>TxSyncLen,x0
        move x0,X:<TxStateCounter
TxDoSync                        ;send symbol-sync. sequence
        move X:<TxStateCounter,a
        tst a #>1,x0
        jle <TxStartPreData
        sub x0,a
        move a,X:<TxStateCounter
        clr a #FFTbuff,r4       ;set all FFT bins to 0
        move #<WindowLen-1,m4
        .loop #WindowLen
          move a,L:(r4)+
        .endl
        move #TxTuneVect,r5     ;copy tune vectors into the FFT buffer
        move #<TuneCarriers-1,m5
        move #FFTbuff+FirstTuneCarr,r4
        move #<TuneCarrSepar,n4
        .loop #TuneCarriers
          move L:(r5),ab        ;get I/Q
          jset #0,r4,TxSync_cont ;flip even carrier's phase
            neg a                ;to make flipping phase wave on the output
            neg b
TxSync_cont move ab,L:(r5)+     ;save the I/Q
          move ab,L:(r4)+n4     ;put it into the FFT buffer
        .endl
        jmp <TxDoFFT

TxStartPreData
        move #>TxDoPreData,x0
        move x0,X:<TxStateRoutine
        move #>TxPreData,x0
        move x0,X:<TxStateCounter

        move #TxDataIniVect,r4
        move #<DataCarriers-1,m4
        move #TxDataVect,r5
        move #<DataCarriers-1,m5
        .loop #DataCarriers             ;copy initial-phase vectors
           move L:(r4)+,ab              ;into the TxDataVect buffer
           move ab,L:(r5)+
        .endl
      if Interleave
        clr a #TxInlvPipe,r0            ;clear the interleave pipe
        move #<Interleave*DataCarriers-1,m0
        .loop #Interleave*DataCarriers
          move a,Y:(r0)+
        .endl
      endif
TxDoPreData
        move X:<TxStateCounter,a
        tst a #>1,x0
        jle <TxStartData
        sub x0,a
        clr a a,X:<TxStateCounter
        move a,L:<TxWord
        jmp <TxEncodeWord

TxStartData
        bclr #0,X:<TxState              ;clear the PTT-ON request
        move #>TxDoData,x0
        move x0,X:<TxStateRoutine
      if SPY
        move #>100,x0                   ;make 100 data symbols
        move x0,X:<TxStateCounter       ;- just to send something
      endif
TxDoData
      if SPY
        move X:<TxStateCounter,a
        tst a #>1,x0
        jle <TxStartPostData
        sub x0,a
        move a,X:<TxStateCounter
;        jsr <Rand48
        move X:<TxStateCounter,x0
        move #>50,x1
        mpy x0,x1,a
        move x0,a1
        move a10,L:<TxWord
      else
        jset #1,X:<TxState,TxStartPostData      ;stop transmiting data
                                                ;when PTT-OFF was requested
        jsr <GetBitBatch
        move a1,X:<TxWord
        jsr <GetBitBatch
        move a1,Y:<TxWord
      endif

TxEncodeWord
      if FEC==1
        move X:<TxWord,a1
        jsr <FEC1511Encode
        move a1,X:<TxWord
        move Y:<TxWord,a1
        jsr <FEC1511Encode
        move a1,Y:<TxWord
      endif

      if FEC==3
        move X:<TxWord,a1
        jsr <WalshEncode
        move a1,X:<TxWord
        move Y:<TxWord,a1
        jsr <WalshEncode
        move a1,Y:<TxWord
      endif

      if Interleave
        move X:<TxWord,a1
        jsr <TxInlv
        move a1,X:<TxWord
        move Y:<TxWord,a1
        jsr <TxInlv
        move a1,Y:<TxWord
      endif

        clr a #FFTbuff,r4       ;set all FFT bins to 0
        move #<WindowLen-1,m4
        .loop #WindowLen
          move a,L:(r4)+
        .endl
;now we encode the di-bits onto the carriers and copy the I/Q vectors
;into the FFT buffer.
;the following di-bit encoding is used: (di-bit -> phase change)
;11 -> 0, 10 -> +90, 00 -> +180, 01 -> +270
        move #TxDataVect,r5
        move #<DataCarriers-1,m5
        move #FFTbuff+FirstDataCarr,r4
        move #<DataCarrSepar,n4
        .loop #DataCarriers
          move L:(r5),ab        ;get I/Q
          jset #0,X:<TxWord,TxBitLow
            neg a               ;if higher bit = 0
            neg b               ;turn by +180
            bchg #0,Y:<TxWord   ;negate the lower bit (for Grey-code)
TxBitLow  jset #0,Y:<TxWord,TxBitLoop_cont
            tfr b,a a,x0        ;if lower bit = 0
            neg a  x0,b         ;turn by +90
TxBitLoop_cont
          jclr #0,r4,TxData_cont        ;extra phase flip for odd carriers
            neg a
            neg b
TxData_cont move ab,L:(r5)+    ;save the I/Q
          move ab,L:(r4)+n4    ;put it into the FFT buffer
          move X:<TxWord,a1
          lsr a1 Y:<TxWord,b1  ;shift the two done bits out
          lsr b1 a1,X:<TxWord
          move b1,Y:<TxWord
        .endl
        jmp <TxDoFFT

TxStartPostData
        bclr #1,X:<TxState      ;clear the PTT-OFF request
        move #>TxDoPostData,x0
        move x0,X:<TxStateRoutine
        move #>TxPostData+(Interleave*DataCarriers+1)/2,x0
        move x0,X:<TxStateCounter
TxDoPostData
      if !SPY
        jset #0,X:<TxState,TxStartData ;if there is a new request for the PTT
                                       ;we append the new data
      endif                            ;to the ongoing transmition
        move X:<TxStateCounter,a
        tst a #>1,x0
        jle <TxStartJam
        sub x0,a
        clr a a,X:<TxStateCounter
        move a,L:<TxWord
        jmp <TxEncodeWord

TxStartJam
        move #>TxDoJam,x0
        move x0,X:<TxStateRoutine
        move #>TxJamLen,x0
        move x0,X:<TxStateCounter
TxDoJam                                 ;send the jamming sequence
        move X:<TxStateCounter,a
        tst a #>1,x0
        jle <TxStartBuffFlush
        sub x0,a
        move a,X:<TxStateCounter

        jsr <Rand48                     ;make a random word
        move a10,L:<TxWord              ;for jamming

        clr a #FFTbuff,r4       ;set all FFT bins to 0
        move #<WindowLen-1,m4
        .loop #WindowLen
          move a,L:(r4)+
        .endl
        move #FFTbuff+FirstDataCarr,r4
        move #<DataCarrSepar,n4
        move #TxDataVect,r5
        move #<DataCarriers-1,m5
        move L:<Vect45,y
        .loop #DataCarriers
          move L:(r5),x         ;get I/Q
          mpy x1,y1,a           ;turn by 45 degrees
          macr -x0,y0,a         ;for maximum phase error
          mpy x1,y0,b           ;at the receiver
          macr x0,y1,b
          jclr #0,X:<TxWord,TxJam_cont1   ;randomly turn by 90 and/or 180
            tfr b,a a,x0
            neg a x0,b
TxJam_cont1 jclr #0,Y:<TxWord,TxJam_cont2
            neg a
            neg b
TxJam_cont2 move ab,L:(r5)+     ;save the I/Q
          move ab,L:(r4)+n4     ;put it into the FFT buffer
          move X:<TxWord,a1
          lsr a1 Y:<TxWord,b1   ;shift the two done bits out
          lsr b1 a1,X:<TxWord
          move b1,Y:<TxWord
        .endl
        jmp <TxDoFFT

TxStartBuffFlush
        move #>TxDoBuffFlush,x0
        move x0,X:<TxStateRoutine
        move #>(BufLen/(SymbolLen*DecimateRatio)+2),x0
        move x0,X:<TxStateCounter
TxDoBuffFlush                           ;let the CODEC's buffer flush
        move X:<TxStateCounter,a        ;the output samples before
        tst a #>1,x0                    ;we turn off the PTT
        jle <TxStartIdle
        sub x0,a
        move a,X:<TxStateCounter
        jmp <TxSilence

TxDoFFT  jsr <DoFFT             ;execute IFFT

TxDoWindow                      ;overlap the new window
        move #FFTbuff,r0
        move #<0,m0
        move #WindowLen/2,n0
        move X:<TxWindow,r5
        move #<WindowLen-1,m5
        move X:<WindowOutTapPtr,r1 ;r1 to address the output window tap
        move #<WindowLen-1,m1
        move X:(r0),x0 Y:(r5)+,y0
        .loop #WindowLen/2
          move L:(r1),ab
          macr x0,y0,a Y:(r0)+n0,x0
          macr x0,y0,b X:(r0),x0 Y:(r5)+,y0
          move ab,L:(r1)+
        .endl
        .loop #WindowLen/2-1
          mpyr x0,y0,a Y:(r0)+n0,x0
          mpyr x0,y0,b X:(r0),x0 Y:(r5)+,y0
          move ab,L:(r1)+
        .endl
        mpyr x0,y0,a Y:(r0)+n0,x0
        mpyr x0,y0,b a,X:(r1)
        move b,Y:(r1)+

        move #AliasFilterOutI,r4
        move #<AliasFilterLen-1,m4
        move #<AliasFilterOutQ-AliasFilterOutI,n4
        move (r2)+
      if RightChannel
        move (r2)+
      endif
        .loop #SymbolLen        ;output interpolation
          move #4*(AliasFilterLen-DecimateRatio),n2
          move L:(r1)+,ab       ;a,b = processed samples (I/Q) ready to be output
          move (r2)-n2            ;move back in the CODEC input buffer
          move #<4,n2             ;n2=4 for easier input adressing
          move a,y1
          move b,y0
          .loop #AliasFilterLen-DecimateRatio
            move X:(r4),x0 Y:(r2),a
            mac x0,y1,a X:(r4+n4),x0
            macr x0,y0,a (r4)+
            move a,Y:(r2)+n2      ;output to the left channel
          .endl
          .loop #DecimateRatio
            move X:(r4),x0
            mpy x0,y1,a X:(r4+n4),x0
            macr x0,y0,a (r4)+
            move a,Y:(r2)+n2
          .endl
          nop
        .endl
        move (r2)-
      if RightChannel
        move (r2)-
      endif
        move r2,X:<TxCodecPtr
        move r1,X:<WindowOutTapPtr

        jmp     <ProcessLoop

CheckSampleBlock        ;a = how many samples we want
        asl a           ;r2 = where we wait for the block to start
        asl a #>2,x1
        add x1,a
        move a,x1
CheckSampleBlock_1
        move r7,a
        move r2,x0
        sub x0,a
        jpl <CheckSample_cmp
          move #>BufLen*4,x0
          add x0,a
CheckSample_cmp
        cmp x1,a
        rts             ;on output: a,x0,x1 are modified
                        ;If carry=0 => we are sure that:
                        ;1. there are at least A new input samples at X:(r2)
                        ;   to be read
                        ;2. there are at least A locations at Y:(r2) to
                        ;   be written


WaitSampleBlock         ;wait for a block of samples from the CODEC
                        ;on input: a = how many samples we want
                        ;          r2 = where the block starts
        jsr <CheckSampleBlock
        jcc <WaitSample_ret
WaitSample_loop
        wait            ;wait (at low power) until an interrupt comes
        jsr <CheckSampleBlock_1
        jcs <WaitSample_loop
WaitSample_ret
        rts             ;on output: a,x0,x1 are modified
                        ;We are sure that:
                        ;1. there are at least A new input samples at X:(r2)
                        ;   to be read
                        ;2. there are at least A locations at Y:(r2) to
                        ;   be written

PI      equ     3.14159265358979323846

;this routine computes a cosine/sine pair using the sine ROM
;with a second order (linear+quadrature) approximation between table points
IQ                              ;x0 = angle ( -1 = -PI, +1 = +PI)
;        ori #%00000100,omr      ;enable the sine ROM table
        move #>$80,x1   ;shift out 8 most significant bits
        mpy x0,x1,a  #>$FF,x0
        move x0,m0
        and x0,a     #>$100,x0
        or x0,a      #<$40,n0
        move a1,r0      ;put the 8 most significant bits into r0 with offset = $100
        move a0,y0      ;save the remaining bits in y0
        jclr #23,y0,SinTable_lev2
          move (r0)+
SinTable_lev2
        move Y:(r0+n0),x0       ;x0 = coarse cosine
        move Y:(r0),x1          ;x1 = coarse sine
        mpyr x1,y0,a  #PI/256.0,y1
        tfr x0,a  a,x1
        macr -x1,y1,a           ;a = fine cosine
        mpyr x0,y0,b  Y:(r0),x1
;        andi #%11111011,omr     ;disable the sine ROM table
        tfr x1,b  b,x1
        macr x1,y1,b  #PI*PI/2.0/65536.0,y1  ;b = fine sine
        mpyr y0,y0,a  a,x0
        move a,y0
        mpyr y0,y1,a
        tfr x0,a  a,y1
        macr -x0,y1,a  b,x1     ;a = super fine cosine
        macr -x1,y1,b           ;b = super fine sine
        rts                     ;x,y are modified
                                ;r0,m0,n0 are modified
                                ;maximum error is about 0.7E-6
                                ;execution time 4+64+4 clock cycles
                                ;including "jsr <IQ_lev2" and "rts"

;Phase48 computes the phase of the given I/Q pair
;this routine is not very precise...
Phase48    ;a = I-part (48 bits), b = Q-part (48 bits)
           ;-1.0 <= a < 1.0, -1.0 <= b < 1.0
        tst b #<0,x0    ;if Q-part negative => turn by PI
        .if <mi>
          neg a #<%10000000,x0
          neg b
        .endi
        tst a b0,y0     ;if I-part negative => turn by -PI/2
        .if <mi>
          bset #22,x0
          tfr a,b b1,y1
          tfr y1,a
          neg b y0,a0
        .endi
        cmp b,a a0,y0   ;if Q>I swap them.
        .if <lt>
          tfr b,a a1,y1
          tfr y1,b
          move y0,b0
          bset #21,x0
        .endi
        tst a #1.0/PI,y0       ;normalize
        .while <nn>
          asl b
          asl a
        .endw
        move a,x1
        andi #$FE,ccr   ;divive Q by I
        rep #24
          div x1,b
        tfr x0,a b0,x0  ;x0 = Q div I, a=approx angle
        btst #21,a1
        .if <cs>
          macr -x0,y0,a #<%00100000,y0
          add y0,a #-(1.0/PI-1.0/4.0),y0
        .else
          mac x0,y0,a #(1.0/PI-1.0/4.0),y0
        .endi
        mpyr x0,y0,b
        mpyr x0,x0,b b,y0
        move b,x0
        mac -x0,y0,a
        rts             ;a = the phase (-1.0 => -PI, 1.0 => PI)
                        ;b,x,y are modified

Rand48                          ;a simple random number generator
        move L:<RandInc,x
        move L:<Rand,a
        btst #23,a1
        adc x,a
        move a10,L:<Rand
        rts                     ;returns a pseudo-random number in a
                                ;modifies x

      if FEC==1
FEC1511Encode           ;a = 11 bits (0-10) to be encoded into 15 bits
        clr b #>%11111111111,x0          ;clear extra bits, leave only
        and x0,a #FEC1511EncodeTable,r0  ;these 11 ones which we will encode
        move a1,x1              ;save the input
        move #$FFFF,m0
        .loop #11
          ror a Y:(r0)+,x0      ;take the input bits
          .if <cs>              ;if the bit is a 1
            eor x0,b            ;add corresponding CRC
          .endi
          nop
        .endl
        tfr b,a
        or x1,a
        rts             ;a = 15 bits containing 11 data bits (0-10)
                        ;plus 4 FEC bits (11-14).
                        ;b,x0,x1,r0,m0 are modified (m0 is set to $FFFF)

FEC1511Decode           ;a1 = 15 bits to be decoded/corrected
        move a1,y1      ;save the input
        jsr <FEC1511Encode ;compute the CRC
        eor y1,a #FEC1511DecodeTable,r0 ;see the difference in CRCs
        move a1,x0
        move #>($800000>>11),y0
        mpy x0,y0,a     ;shift the CRC difference into bits 0-3
        move a1,n0
        move y1,a
        move Y:(r0+n0),x0  ;pickup the correction
        eor x0,a x0,b   ;and correct the data
        rts             ;a1 = corrected 11 data bits (0-10)
                        ;b1 = error pattern (bits 0..14)
                        ;plus the old CRC bits (11-14)
                        ;x0,x1,y0,y1,r0,m0,n0 are modified
      endif

      if FEC==3
WalshEncode
        move #>%11111,x0        ;a1 = 5 bits to be encoded
        and x0,a #WalshTable,r0 ;lookup the table and pick up
        move #$FFFF,m0          ;the element pointed by the input 5 bits
        move a1,n0
        nop
        move Y:(r0+n0),a1
        rts                     ;a1 = 15-bit codeword to be transmitted
                                ;x0,r0/m0/n0 are modified

WalshDecode                     ;a1 = 15-bit codeword to be decoded
        move a1,y1              ;save the input
        move #<0,x1             ;x1 = 0 for fast bit counting
        move #>16,y0            ;y0 = initial distance
        move #WalshTable,n0     ;here we use a brutal force approach:
        move #$FFFF,m0          ;we look through all the 32 codewords
        move #WalshTable+32,r0  ;and pick up the one with least number
        .loop #32               ;of different bits
          clr b Y:-(r0),a1      ;get the codeword, clear the bit counter
          eor y1,a x1,x0        ;make the difference, clear x1:x0
          .loop #15             ;count in b the bits which are equal to 1
            lsr a1
            adc x,b
          .endl
          tfr y0,b b0,x0
          cmp x0,b
          tge x0,b r0,r1
          move b,y0
        .endl
        move r1,r0
        nop
        move Y:(r0)-n0,b
        eor y1,b r0,a1
        rts                     ;a1 = 5 decoded bits (0..4)
                                ;b1 = error pattern (bits 0..14)
                                ;x0,x1,y0,y1,r0,m0,n0,r1 are modified
                                ;if n0=0 => there was no error

      endif

        if !SPY

; KISS control frame handling - called by LEONID when a KISS control-type
; frame is received with non-standard parameters
KISSctrl rts

; transmitter PTT control - a routine called by LEONID to say that we
; should transmit (carry=1) or stop transmitting (carry=0)
; we don't turn the PTT on/off here
;the modem will do it when it's the right time
PTTctrl jcc <PTT_off
        bset #0,X:<TxState      ;set the PTT-on request flag
        rts
PTT_off bset #1,X:<TxState      ;set the PTT-off request flag
        rts

GetBitBatch             ;get a batch of bits to be transmitted
        .loop #BitBatchLen
          getbit
          move X:<Scratch,a1
          ror a1
          move a1,X:<Scratch
        .endl
        rep #24-BitBatchLen
          lsr a1
        rts             ;a1 = a batch of bits given by LEONID's HDLC encoder
                        ;a,b,x,y,r0/m0/n0 are possibly modified
                        ;(getbit modifies them)

PutBitBatch                     ;a1 = a batch of bits to be passed
        .loop #BitBatchLen      ;to the LEONID's HDLC decoder
           lsr a1
           move a1,X:<Scratch
           putbit
           move X:<Scratch,a1
        .endl
        rts                     ;possibly modifies a,b,x,y,r0/m0/n0
                                ;(putbit uses these registers)
        endif

        if Interleave

TxInlv                  ;input: a1 = bit-batch with new data
        move a1,x1
        move #InterleavePattern,r1
        move #$FFFF,m1
        move X:<TxInlvPtr,r0
        move #<Interleave*DataCarriers-1,m0
        move #<Interleave,n0
        .loop #DataCarriers
          tfr x1,a Y:(r1)+,x0
          and x0,a Y:(r0),y0
          or y0,a 
          move a1,Y:(r0)+n0
        .endl
        clr b Y:(r0),a1
        move b,Y:(r0)+
        move r0,X:<TxInlvPtr
        rts             ;output: a1 = bit-batch to be transmitted
                        ;b,x0,x1,y0, r0/n0/m0, r1/m1 are modified

RxDeInlv                ;input: a1 = received bit-batch
        move #InterleavePattern,r1
        move #$FFFF,m1
        move X:<RxInlvPtr,r0
        move #<Interleave*DataCarriers-1,m0
        move a1,x1
        clr a #<Interleave,n0
        .loop #DataCarriers
          move Y:(r0)+n0,b1
          move Y:(r1)+,x0
          and x0,b
          move b1,y0
          or y0,a
        .endl
        move x1,Y:(r0)+
        move r0,X:<RxInlvPtr
        rts             ;output: a1 = de-interleaved bit-batch
                        ;b,x0,x1,y0, r0/m0/n0, r1/m1 are modified
        endif

        if RxAGC

RxGainUp                        ;increase the CODEC's input gain (both channels)
        clr a #>$0F0F00,x0      ;r2/m2 should point to the CODEC's buffer
        move Y:(r2),a1          ;get the CODEC's input control word
        and x0,a                ;extract the gain bits
	cmp x0,a  #>$010100,x0  ;already maximum ?
        jeq <RxGainChange_abort ;if so then abort this attempt
	add x0,a  #>$F0F000,x0  ;if not, increment the gain by 1
	move a1,x1
	move Y:(r2),a1          ;and reload all the control words
	and x0,a  n2,x0         ;in the output buffer
	or x1,a  #<4,n2         ;make n2=4 for a moment
	.loop #BufLen
	  move a1,Y:(r2)+n2
	.endl
	move x0,n2              ;restore n2
        andi #$FE,ccr           ;clear carry
        rts                     ;modifies a,x0,x1

RxGainDown                      ;decrease the CODEC's input gain (both channels)
        clr a #>$0F0F00,x0      ;r2/m2 should point to the CODEC's buffer
        move Y:(r2),a1          ;get the CODEC's input control word
        and x0,a  #>$010100,x0  ;extract the gain bits
        sub x0,a  #>$F0F000,x0  ;attempt to decrease the gain
        jcs <RxGainChange_abort ;jump if overflow
        move a1,x1
        move Y:(r2),a1          ;reload all the input control words
        and x0,a  n2,x0         ;in the buffer with the new input gain
        or x1,a  #<4,n2         ;n2=4 for a moment
        .loop #BufLen
          move a1,Y:(r2)+n2
        .endl
        move x0,n2              ;restore n2
        andi #$FE,ccr           ;clear carry
        rts                     ;modifies a,x0,x1

RxGainChange_abort
        ori #$01,ccr            ;set carry to indicate that we couldn't
        rts                     ;change the gain

        endif

Initialize      ;initialize registers, buffers, windows, etc.

        ori #%00000100,omr      ;enable the sine ROM table
                                ;this is for the IQ routine

        clr a #WindowInpTap,r0  ;clean the window taps
        move #<WindowLen-1,m0
        rep #WindowLen
          move a,L:(r0)+
        move #WindowOutTap,r0
        rep #WindowLen
          move a,L:(r0)+

        clr a #RxPipe,r0        ;clear the receiver pipe
        move #<DataCarriers*RxPipeLen-1,m0
        rep #DataCarriers*RxPipeLen
          move a,L:(r0)+

        move #$180,r0           ;initialize the sine/cosine table
        move #<$FF,m0           ;by taking every Nth value from the on-chip
        move #<$100/WindowLen,n0 ;ROM sine table (N=256/WindowLen)
        move #FFTcoef,r4         ;This works only for FFT lengths up to 256
        move #<WindowLen-1,m4     ;note that the table for the FFT starts
        move #<3*(WindowLen/4),n4 ;with cos=-1.0 and sin=0.0
        .loop #WindowLen          ;X memory contains the cosine
          move Y:(r0)+n0,a        ;Y memory contains the sine
          move a,X:(r4+n4)
          move a,Y:(r4)+
        .endl

      if !SPY
        move #>1,a              ;minimal TxDelay and TxTail:
        move a,p:kiss_pars      ;we make the delays by ourselfs
        move a,p:kiss_pars+3    ;in the Tx state machine

        if TxPTThack            ;if PTT-hack
          move #>1,a
          move a,p:kiss_pars+4  ;set full duplex to disable LEONID's
        endif                   ;slot-time/persistance

        move #KISSctrl,a1       ;switch serial interface to KISS mode
        move #PTTctrl,b1
        opensc
      endif

        move #Buffer+2,r7        ;for the CODEC's interrupt routine
        move #BufLen*4-1,m7

        move #Buffer,r2          ;for "cdctrl" to initialize the buffer
        move #<4-1,n2
        move #BufLen*4-1,m2
                        ;initialize input/output control words in the buffer
                        ;zero input/output data
      if EVM56K         ;for EVM56002 use MIC input
        ctrlcd  1,r2,BufLen,MIC,RxGain,RxGain,LINEO|HEADP,TxAttenuate,TxAttenuate
      else              ;for DSPCARD4 use LINE input
        ctrlcd  1,r2,BufLen,LINEI,RxGain,RxGain,LINEO|HEADP,TxAttenuate,TxAttenuate
      endif
        opencd SampleFreq/1000.0,HPF     ;start taking samples at given rate

        jmp <ProcessLoop

        if SPY

SpySync move a10,L:<SpySave     ;output: carry=1 => no spy request
        move a2,X:<SpySave+1    ;carry=0 => spy request !
        move x0,Y:<SpySave+1    ;512 words (jsr <SpyA) should follow
        move x1,Y:<SpyCount
        move X:<SpyCount,a
        tst a
        jne <SpyCont
        lookc 0
        jcs <Spy_end
        move #>'S',a
        cmp x0,a
        ori #$01,ccr
        jne <Spy_end
        move #>'P',x0
        putc
        move #>512,a
        move a,X:<SpyCount
SpyCont andi #$FE,ccr
        jmp <Spy_end

SpyA    move a10,L:<SpySave
        move a2,X:<SpySave+1
        move x0,Y:<SpySave+1
        move x1,Y:<SpyCount
        move X:<SpyCount,a
        tst a
        jne <Spy_copy

Spy_check
        lookc 0
        jcs <Spy_end
        move #>'S',a
        cmp x0,a
        jne <Spy_end
        move #>'P',x0
        putc
        move #>512,a
Spy_copy
        move #>1,x0
        sub x0,a
        move a,X:<SpyCount

        move X:<SpySave,a
        rep #8
          lsr a
        move a1,x0
	putc
        move X:<SpySave,a
        rep #16
          lsr a
        move a1,x0
        putc

Spy_end move L:<SpySave,a10
        move X:<SpySave+1,a2
        move Y:<SpySave+1,x0
        move Y:<SpyCount,x1
        rts

        endif

;*************************************************************************
;Internal data RAM

        LOMEM X:$0000,Y:$0000,L:$0000
        HIMEM X:$00FF,Y:$00FF,L:$00FF

        org L:user_data

        if SPY
SpySave dc 0,0
SpyCount dc 0
        endif

TxWord  ds 1    ;holds the 30-bit word to be transmitted
Vect45  ds 1    ;45 degree vector

RxWord  ds 1    ;holds the Rx word just received
RxTmpI  ds 1    ;temporary storages for the receiver's symbol decoder
RxTmpQ  ds 1

Scratch ds 1    ;general purpose scratch variable

Rand    dc $954820584783        ;seed and increment for the random
RandInc dc $584927603412        ;number generator

LastL = *
        org X:LastL
        org Y:LastL

        org X:
WindowInpTapPtr dc WindowInpTap
WindowOutTapPtr dc WindowOutTap
RxTimeToSlide dc SymbolLen/2
RxCodecPtr    dc Buffer
TxCodecPtr    dc Buffer

RxCarPhase    dc 0      ;Rx mixer carrier phase and initial frequency
RxCarFreq     dc 0

RxPhaseCorr dc 0
RxPipePtr   dc RxPipe

RxWindow    dc ToneWindowInp    ;Rx FFT window in use at the moment
TxWindow    dc ToneWindowOut    ;Tx FFT window in use at the moment

TxStateRoutine  dc TxStartIdle
TxStateCounter  dc 0

RxState         dc 0            ;bit #0 = 1 => at-symbol sample
                                ;bit #1 = 1 => data correl. update allowed
                                ;bit #2 = 1 => channel is busy (DCD is on)

TxState         dc 0            ;bit #0 = 1 => PTT ON was requested
                                ;bit #1 = 1 => PTT OFF was requested

RxStateRoutine  dc RxStartIdle
RxStateCounter  dc 0
RxAcceptCount   dc 0

        if FEC
RxErrPattern    dc 0            ;FEC error pattern
        endif

        if RxAGC
RxAudioPeak     dc 0            ;peak level of the input audio
RxAudioMS       dc 0            ;Mean Square (the power) of the input audio
RxAGCcount      dc RxAGCfall
        endif

        if Interleave
RxInlvPtr dc RxInlvPipe
TxInlvPtr dc TxInlvPipe
        endif

        if RxCtrlUpDown
RxUpDownTime dc 0
        endif

        org Y:
RxTunePhase dsm TuneCarriers
RxSyncDelay dsm TuneCarriers

        org X:
LastX = *
        org Y:
LastY = *
        if @cvi(LastX)>=@cvi(LastY)
          org L:LastX
        else
          org L:LastY
        endif

RxTuneCorrel dsm TuneCarriers*8 ;various correlations for
                                ;the tuning/sync. phase

RxDataCorrel   dsm 3*DataCarriers ;various correlations for the data phase

        org X:Vect45
        dc @sqt(0.5)
;        dc 0.6
        org Y:Vect45
        dc @sqt(0.5)
;        dc 0.6
;*************************************************************************
;External data RAM

        if EVM56K
          LOMEM X:$2000,Y:$0100,L:$2000
          HIMEM X:$3FFF,Y:$3FFF,L:$3FFF
        else
          LOMEM X:$0100,Y:$0100,L:$0100
          HIMEM X:$1FFF,Y:$3FFF,L:$1FFF
        endif

      if EVM56K
        org L:$2000
      else
        org L:$200
      endif

        org L:
WindowInpTap dsm WindowLen
WindowOutTap dsm WindowLen
FFTcoef dsm WindowLen   ;sine/cosine table for the FFT and other stuff
FFTbuff dsm WindowLen   ;Work buffer for the FFT routine

RxPipe  dsm DataCarriers*RxPipeLen      ;here we store the FFT results
                                        ;for the tone we listen to.
                                        ;this is not large so we could place
                                        ;it in internal RAM...
TxDataIniVect  dsm DataCarriers         ;init. vectors (amplitude and phase)
                                        ;for the Tx should be made such that
                                        ;the Peak/RMS is minimal
TxDataVect     dsm DataCarriers         ;these are used during data
                                        ;transmition phase

TxTuneIniVect  dsm TuneCarriers         ;init. vectors for tuning preamble
TxTuneVect     dsm TuneCarriers

LastL = *
        org X:LastL
        org Y:LastL

        org Y:
AliasFilterInpI dsm AliasFilterLen      ;anti-alias input filter (I/Q)
AliasFilterInpQ dsm AliasFilterLen

        org X:
AliasFilterOutI dsm AliasFilterLen      ;anti-alias output filter (I/Q)
AliasFilterOutQ dsm AliasFilterLen

        org Y:
ToneWindowInp   dsm WindowLen           ;input/output FFT window
ToneWindowOut   dsm WindowLen           ;for pure tone Tx/Rx
DataWindowInp   dsm WindowLen           ;input/output FFT window
DataWindowOut   dsm WindowLen           ;for data symbols Tx/Rx

      if Interleave
        org Y:
RxInlvPipe      dsm Interleave*DataCarriers
TxInlvPipe      dsm Interleave*DataCarriers
      endif

      if Interleave
        org Y:
InterleavePattern
       if ScrambledInterleave
        dc %000000000000001
        dc %000000000010000
        dc %000000100000000
        dc %001000000000000
        dc %000000000000010
        dc %000000000100000
        dc %000001000000000
        dc %010000000000000
        dc %000000000000100
        dc %000000001000000
        dc %000010000000000
        dc %100000000000000
        dc %000000000001000
        dc %000000010000000
        dc %000100000000000
       else                     ;linear interleave
        dc %000000000000001
        dc %000000000000010
        dc %000000000000100
        dc %000000000001000
        dc %000000000010000
        dc %000000000100000
        dc %000000001000000
        dc %000000010000000
        dc %000000100000000
        dc %000001000000000
        dc %000010000000000
        dc %000100000000000
        dc %001000000000000
        dc %010000000000000
        dc %100000000000000
       endif
      endif

      if FEC==1
        org Y:
FEC1511EncodeTable
        dc %001100000000000
        dc %010100000000000
        dc %011000000000000
        dc %011100000000000
        dc %100100000000000
        dc %101000000000000
        dc %101100000000000
        dc %110000000000000
        dc %110100000000000
        dc %111000000000000
        dc %111100000000000

FEC1511DecodeTable
        dc %000000000000000
        dc %000100000000000
        dc %001000000000000
        dc %000000000000001
        dc %010000000000000
        dc %000000000000010
        dc %000000000000100
        dc %000000000001000
        dc %100000000000000
        dc %000000000010000
        dc %000000000100000
        dc %000000001000000
        dc %000000010000000
        dc %000000100000000
        dc %000001000000000
        dc %000010000000000
      endif

      if FEC==3

        org Y:
WalshTable
        dc %000000000000000     ;these are 32 codewords which are 15 bit long.
        dc %000000001111111     ;Every codeword differs from any other
        dc %000011110000111     ;at 7 or more bits
        dc %000011111111000     ;thus at the receiver we can correct
        dc %001100110011001     ;up to 3 bit errors
        dc %001100111100110
        dc %001111000011110
        dc %001111001100001
        dc %010101010101010
        dc %010101011010101
        dc %010110100101101
        dc %010110101010010
        dc %011001100110011
        dc %011001101001100
        dc %011010010110100
        dc %011010011001011
        dc %100101100110100
        dc %100101101001011
        dc %100110010110011
        dc %100110011001100
        dc %101001010101101
        dc %101001011010010
        dc %101010100101010
        dc %101010101010101
        dc %110000110011110
        dc %110000111100001
        dc %110011000011001
        dc %110011001100110
        dc %111100000000111
        dc %111100001111000
        dc %111111110000000
        dc %111111111111111

      endif

        org X:
LastX = *
        org Y:
LastY = *
        if @cvi(LastX)>=@cvi(LastY)
          org L:LastX
        else
          org L:LastY
        endif

Buffer  dsm BufLen*4    ;CODEC's input/output buffer

;*************************************************************************
;constant tables: FIR and window shapes, FFT coefficiants, etc.

        nolist
        include 'newqpsk.dat'
        list

;*************************************************************************

        end

